<?php

namespace App\Console\Commands;

use App\Models\Side;
use Illuminate\Console\Command;

class LibrariesXBSync extends Command
{
    private const DADES_OBERTES_BIBLIOTEQUES_ENDPOINT = "https://do.diba.cat/api/dataset/biblioteques/format/json";

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:libraries-xb-sync';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Synchronize libraries of the XB';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $librariesFromOpenSourceData = file_get_contents(self::DADES_OBERTES_BIBLIOTEQUES_ENDPOINT);

        if ($librariesFromOpenSourceData === false) {
            return $this->info("Unable to read libraries of the XB");
        }

        $librariesArray = json_decode($librariesFromOpenSourceData, true);

        $librariesElements = $librariesArray["elements"];

        foreach ($librariesElements as $library) {
            $winterHours = [
                'dilluns' => $library['horari_hivern_dilluns'],
                'dimarts' => $library['horari_hivern_dimarts'],
                'dimecres' => $library['horari_hivern_dimecres'],
                'dijous' => $library['horari_hivern_dijous'],
                'divendres' => $library['horari_hivern_divendres'],
                'dissabte' => $library['horari_hivern_dissabte'],
                'diumenge' => $library['horari_hivern_diumenge'],
            ];

            $summerHours = [
                'dilluns' => $library['horari_estiu_dilluns'],
                'dimarts' => $library['horari_estiu_dimarts'],
                'dimecres' => $library['horari_estiu_dimecres'],
                'dijous' => $library['horari_estiu_dijous'],
                'divendres' => $library['horari_estiu_divendres'],
                'dissabte' => $library['horari_estiu_dissabte'],
                'diumenge' => $library['horari_estiu_diumenge'],
            ];

            $side = Side::create([
                'identifier' => $library['punt_id'],
                'name' => $library['adreca_nom'],
                'description' => $library['descripcio'],
                'address' => $library['grup_adreca']['adreca_completa'],
                'postal_code' => $library['grup_adreca']['codi_postal'] ?? null,
                'location' => $library['grup_adreca']['municipi_nom'],
                'lat_long' => $library['localitzacio'],
                'contact_email' => $library['email'][0] ?? null,
                'contact_phone' => $library['telefon_contacte'][0] ?? null,
                'image_url' => $library['imatge'][0] ?? null,
                'winter_start_date' => $this->parseDate($library['inici_horari_hivern']) ?? new \DateTime(),
                'summer_start_date' => $this->parseDate($library['inici_horari_estiu']) ?? new \DateTime(),
                'operating_hours_winter' => $this->transformOperatingHours($winterHours),
                'operating_hours_summer' => $this->transformOperatingHours($summerHours),
            ]);
        }

        return $this->info("Synchronized libraries of the XB finished");
    }

    private function parseDate(string $dateString): ?string
    {
        preg_match('/<span>(\d{1,2}) de ([a-z]+)<\/span>/', $dateString, $matches);

        if (!$matches || count($matches) < 3) {
            return null;
        }

        $day = $matches[1];
        $month = $matches[2];

        $monthsMap = [
            'gener' => 'January',
            'febrer' => 'February',
            'març' => 'March',
            'abril' => 'April',
            'maig' => 'May',
            'juny' => 'June',
            'juliol' => 'July',
            'agost' => 'August',
            'setembre' => 'September',
            'octubre' => 'October',
            'novembre' => 'November',
            'desembre' => 'December',
        ];

        $monthEnglish = $monthsMap[strtolower($month)] ?? null;

        if (!$monthEnglish) {
            return null;
        }

        try {
            return \Carbon\Carbon::createFromFormat('d F', "$day $monthEnglish")->format('Y-m-d H:i:s');
        } catch (\Exception $e) {
            return null;
        }
    }

    private function transformOperatingHours(array $dailyHours)
    {
        $daysOfWeek = [
            'dilluns' => 'Monday',
            'dimarts' => 'Tuesday',
            'dimecres' => 'Wednesday',
            'dijous' => 'Thursday',
            'divendres' => 'Friday',
            'dissabte' => 'Saturday',
            'diumenge' => 'Sunday',
        ];

        $formattedHours = [];

        foreach ($dailyHours as $day => $hoursText) {
            $dayName = $daysOfWeek[$day] ?? null;

            if (!$dayName) {
                continue;
            }

            if (strtolower($hoursText) === 'tancat') {
                $formattedHours[] = [
                    'day' => $dayName,
                    'open_time' => null,
                    'close_time' => null,
                ];
            } else {
                $timeIntervals = explode(' i ', strtolower($hoursText));
                foreach ($timeIntervals as $interval) {
                    if (preg_match('/de (\d{1,2}\.\d{2}) a (\d{1,2}\.\d{2})/', $interval, $matches)) {
                        $openTime = str_replace('.', ':', $matches[1]) . ':00';
                        $closeTime = str_replace('.', ':', $matches[2]) . ':00';

                        $formattedHours[] = [
                            'day' => $dayName,
                            'open_time' => $openTime,
                            'close_time' => $closeTime,
                        ];
                    }
                }
            }
        }

        return $formattedHours;
    }
}
