<?php

namespace App\Filament\Resources;

use App\Filament\Resources\SideResource\Pages;
use App\Filament\Resources\SideResource\RelationManagers;
use App\Models\Enum\BookingStatusEnum;
use App\Models\Enum\RoleEnum;
use App\Models\Side;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Support\Enums\ActionSize;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class SideResource extends Resource
{
    protected static ?string $model = Side::class;

    protected static ?string $navigationGroup = "Library Management";

    protected static ?string $navigationIcon = 'heroicon-o-building-library';

    public static function canViewAny(): bool
    {
        return auth()->user()->role_id !== RoleEnum::ID[RoleEnum::SIDE_USER];
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('identifier')
                    ->required()
                    ->maxLength(255),

                Forms\Components\TextInput::make('name')
                    ->required()
                    ->maxLength(255),

                Forms\Components\TextInput::make('description')
                    ->nullable()
                    ->maxLength(255),

                Forms\Components\TextInput::make('address')
                    ->required()
                    ->maxLength(255),

                Forms\Components\TextInput::make('postal_code')
                    ->nullable()
                    ->maxLength(255),

                Forms\Components\TextInput::make('location')
                    ->nullable()
                    ->maxLength(255),

                Forms\Components\TextInput::make('lat_long')
                    ->nullable()
                    ->maxLength(255),

                Forms\Components\TextInput::make('contact_email')
                    ->nullable()
                    ->maxLength(255),

                Forms\Components\TextInput::make('contact_phone')
                    ->nullable()
                    ->maxLength(255),

                Forms\Components\TextInput::make('image_url')
                    ->nullable()
                    ->maxLength(255),

                Forms\Components\DateTimePicker::make('winter_start_date')
                    ->required(),

                Forms\Components\DateTimePicker::make('summer_start_date')
                    ->required(),

                Forms\Components\Repeater::make('operating_hours_winter')
                    ->schema([
                        Forms\Components\Select::make('day')
                            ->label('Day of the Week')
                            ->options([
                                'Monday' => 'Monday',
                                'Tuesday' => 'Tuesday',
                                'Wednesday' => 'Wednesday',
                                'Thursday' => 'Thursday',
                                'Friday' => 'Friday',
                                'Saturday' => 'Saturday',
                                'Sunday' => 'Sunday',
                            ]),

                        Forms\Components\TimePicker::make('open_time')
                            ->label('Opening Time'),

                        Forms\Components\TimePicker::make('close_time')
                            ->label('Closing Time'),
                    ])
                    ->columns(1)
                    ->minItems(0)
                    ->maxItems(7)
                    ->defaultItems(1)
                    ->addActionLabel('Add Day'),

                Forms\Components\Repeater::make('operating_hours_summer')
                    ->schema([
                        Forms\Components\Select::make('day')
                            ->label('Day of the Week')
                            ->options([
                                'Monday' => 'Monday',
                                'Tuesday' => 'Tuesday',
                                'Wednesday' => 'Wednesday',
                                'Thursday' => 'Thursday',
                                'Friday' => 'Friday',
                                'Saturday' => 'Saturday',
                                'Sunday' => 'Sunday',
                            ]),

                        Forms\Components\TimePicker::make('open_time')
                            ->label('Opening Time'),

                        Forms\Components\TimePicker::make('close_time')
                            ->label('Closing Time'),
                    ])
                    ->columns(1)
                    ->minItems(0)
                    ->maxItems(7)
                    ->defaultItems(1)
                    ->createItemButtonLabel('Add Day'),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('identifier')
                    ->searchable()
                    ->sortable()
                    ->label('Identifier'),

                Tables\Columns\TextColumn::make('name')
                    ->searchable()
                    ->sortable()
                    ->label('Name'),

                Tables\Columns\TextColumn::make('contact_email')
                    ->label('Contact Email')
                    ->searchable()
                    ->sortable(),

                Tables\Columns\TextColumn::make('address')
                    ->label('Dirección'),

                Tables\Columns\TextColumn::make('computers_count')
                    ->label('Computers Count')
                    ->getStateUsing(function ($record) {
                        return $record->computers()->count();
                    })
                    ->sortable(),

                Tables\Columns\TextColumn::make('active_bookings_count')
                    ->label('Active Bookings')
                    ->getStateUsing(function ($record) {
                        return $record->bookings()
                            ->where("booking_status_id", BookingStatusEnum::ID[BookingStatusEnum::CONFIRMED])
                            ->count()
                        ;
                    })
                    ->sortable(),
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Tables\Actions\EditAction::make(),

                    Tables\Actions\Action::make('go_to_computers')
                        ->label('Go To Computers')
                        ->url(fn ($record) => "/admin/computers?tableFilters[side_id][value]=". $record->id)
                        ->icon('heroicon-o-computer-desktop')
                        ->color('secondary'),

                    Tables\Actions\Action::make('go_to_active_bookings')
                        ->label('Go To Active Bookings')
                        ->url(fn ($record) => "/admin/bookings?tableFilters[side_id][value]=". $record->id)
                        ->icon('heroicon-o-calendar-date-range')
                        ->color('success'),
                ])
                ->label('Actions')
                ->icon('heroicon-m-ellipsis-vertical')
                ->size(ActionSize::Small)
                ->color('primary')
                ->button()
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->searchable()
        ;
    }

    public static function getRelations(): array
    {
        return [

        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListSides::route('/'),
            'create' => Pages\CreateSide::route('/create'),
            'edit' => Pages\EditSide::route('/{record}/edit'),
        ];
    }
}
