<?php

namespace App\Filament\Resources;

use App\Filament\Resources\UserTypeRuleResource\Pages;
use App\Filament\Resources\UserTypeRuleResource\RelationManagers;
use App\Models\Enum\RoleEnum;
use App\Models\UserType;
use App\Models\UserTypeRule;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class UserTypeRuleResource extends Resource
{
    protected static ?string $model = UserTypeRule::class;

    protected static ?string $navigationGroup = "User Management";

    protected static ?string $navigationIcon = 'heroicon-o-document-duplicate';

    public static function canViewAny(): bool
    {
        return auth()->user()->role_id !== RoleEnum::ID[RoleEnum::SIDE_USER];
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('user_type_id')
                    ->label('User Type')
                    ->options(function () {
                        return UserType::all()->pluck('name', 'id');
                    })
                    ->required()
                    ->searchable(),

                Forms\Components\TextInput::make('name')
                    ->label('Name')
                    ->required()
                    ->maxLength(255),

                Forms\Components\TextInput::make('description')
                    ->label('Description')
                    ->nullable(),

                Forms\Components\TextInput::make('parameter')
                    ->label('Parameter')
                    ->nullable()
                    ->maxLength(255),

                Forms\Components\TagsInput::make('parameter_value')
                    ->label('Parameter Value')
                    ->nullable()
                    ->placeholder('Add values, domains, apps.'),

                Forms\Components\DateTimePicker::make('active_from_date')
                    ->label('Active From Date')
                    ->nullable(),

                Forms\Components\DateTimePicker::make('deactivate_date')
                    ->label('Deactivate Date')
                    ->nullable(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->label('Rule Name')
                    ->sortable()
                    ->searchable()
                    ->limit(50),

                Tables\Columns\TextColumn::make('description')
                    ->label('Description')
                    ->limit(),

                Tables\Columns\TextColumn::make('userType.name')
                ->label('User Type')
                    ->sortable()
                    ->searchable(),

                Tables\Columns\TextColumn::make('parameter')
                    ->label('Parameter')
                    ->limit(50)
                    ->sortable(),

                Tables\Columns\TextColumn::make('parameter_value')
                    ->label('Parameter Value')
                    ->limit()
                    ->sortable(),
            ])
            ->filters([
                Tables\Filters\Filter::make('user_type')
                    ->label('User Type')
                    ->form([
                        Forms\Components\Select::make('user_type_id')
                            ->label('User Type')
                            ->options(function () {
                                return \App\Models\UserType::all()->pluck('name', 'id');
                            })
                            ->searchable()
                            ->placeholder('Select User Type')
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query->when(
                            $data['user_type_id'] ?? null,
                            fn(Builder $query, $userTypeId) => $query->where('user_type_id', $userTypeId)
                        );
                    }),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListUserTypeRules::route('/'),
            'create' => Pages\CreateUserTypeRule::route('/create'),
            'edit' => Pages\EditUserTypeRule::route('/{record}/edit'),
        ];
    }
}
